<?php
require_once 'config.php';
include 'security.php';
require_once 'includes/openai_helper.php';

// Get the logged-in user's ID
$user_id = isset($_SESSION['user_id']) ? intval($_SESSION['user_id']) : 0;

if($user_id <= 0) {
    header('Location: login.php');
    exit();
}

$message = '';
$message_type = '';
$force_manual_mode = false;

// Prefill values for the form (used after AI extraction)
$prefill_merchant = '';
$prefill_amount = '';
$prefill_category = '';
$prefill_account = '';
$prefill_date = date('Y-m-d');
$prefill_notes = '';

// Handle form submission
if($_SERVER['REQUEST_METHOD'] == 'POST') {
    $mode = isset($_POST['mode']) ? $_POST['mode'] : 'manual';

    // --- Upload mode: extract data with OpenAI, then show the form prefilled ---
    if($mode === 'upload') {
        if(empty($_FILES['expense-upload']['tmp_name'])) {
            $message = 'Please choose a file to upload.';
            $message_type = 'error';
        } else {
            $file = $_FILES['expense-upload'];
            if($file['error'] !== UPLOAD_ERR_OK) {
                $message = 'Upload error. Please try again.';
                $message_type = 'error';
            } elseif($file['size'] > 10 * 1024 * 1024) {
                $message = 'File too large. Max 10MB.';
                $message_type = 'error';
            } else {
                $result = extract_transaction_from_file($file['tmp_name'], 'expense');
                if(!empty($result['error'])) {
                    $message = $result['error'];
                    $message_type = 'error';
                    $force_manual_mode = true;
                } else {
                    $message = 'Data extracted. Please review and submit.';
                    $message_type = 'success';
                    $force_manual_mode = true;
                    $prefill_merchant = $result['description'] ?? '';
                    $prefill_amount = $result['amount'] ?? '';
                    $prefill_category = $result['category'] ?? '';
                    $prefill_account = '';
                    $prefill_date = !empty($result['date']) ? $result['date'] : date('Y-m-d');
                    $prefill_notes = '';
                }
            }
        }
    }

    // --- Manual mode: existing save flow ---
    if($mode !== 'upload') {
        $merchant_name = isset($_POST['merchant-name']) ? trim($_POST['merchant-name']) : '';
        $amount = isset($_POST['expense-amount']) ? trim($_POST['expense-amount']) : '';
        $category_input = isset($_POST['expense-category-input']) ? trim($_POST['expense-category-input']) : '';
        $category_id = 0;
        $account_id = isset($_POST['expense-account']) ? intval($_POST['expense-account']) : 0;
        $expense_date = isset($_POST['expense-date']) ? trim($_POST['expense-date']) : '';
        $notes = isset($_POST['expense-notes']) ? trim($_POST['expense-notes']) : '';

        // Keep values in case of validation errors
        $prefill_merchant = $merchant_name;
        $prefill_amount = $amount;
        $prefill_category = $category_input;
        $prefill_account = $account_id;
        $prefill_date = $expense_date ?: $prefill_date;
        $prefill_notes = $notes;

        // Handle category - check if input matches existing category or is new (assisted by AI)
        if(!empty($category_input)) {
            // Check if category exists (case-insensitive)
            $check_cat = $mysql->query("SELECT category_id FROM categories WHERE LOWER(label) = LOWER('" . $mysql->real_escape_string($category_input) . "') AND type = 'expense' AND (user_id = " . $user_id . " OR user_id IS NULL) LIMIT 1");
            if($check_cat && $check_cat->num_rows > 0) {
                $cat_row = $check_cat->fetch_assoc();
                $category_id = intval($cat_row['category_id']);
            } else {
                // Create new category
                $create_cat_sql = "INSERT INTO categories (user_id, label, type) VALUES (" . $user_id . ", '" . $mysql->real_escape_string($category_input) . "', 'expense')";
                if($mysql->query($create_cat_sql)) {
                    $category_id = intval($mysql->insert_id);
                } else {
                    $message = 'Error creating category: ' . $mysql->error;
                    $message_type = 'error';
                }
            }
        }

        // Validate required fields
        if(empty($merchant_name) || empty($amount) || $category_id <= 0 || $account_id <= 0 || empty($expense_date)) {
            $message = 'Please fill in all required fields.';
            $message_type = 'error';
        } else {
            // Clean amount (remove $ and commas)
            $amount_clean = preg_replace('/[^0-9.]/', '', $amount);
            $amount_clean = floatval($amount_clean);

            if($amount_clean <= 0) {
                $message = 'Please enter a valid amount.';
                $message_type = 'error';
            } else {
                // Parse date (assuming format MM/DD/YYYY or YYYY-MM-DD)
                if(strpos($expense_date, '/') !== false) {
                    $date_parts = explode('/', $expense_date);
                    if(count($date_parts) == 3) {
                        $transaction_date = $date_parts[2] . '-' . $date_parts[0] . '-' . $date_parts[1];
                    } else {
                        $transaction_date = date('Y-m-d');
                    }
                } else {
                    $transaction_date = $expense_date;
                }

                // Verify account belongs to user
                $account_check = $mysql->query("SELECT account_id FROM accounts WHERE account_id = " . $account_id . " AND user_id = " . $user_id);
                if($account_check && $account_check->num_rows > 0) {
                    // Insert transaction (expenses are stored as negative amounts)
                    $note_text = $merchant_name . ($notes ? ' - ' . $notes : '');
                    $sql = "INSERT INTO transactions (account_id, category_id, amount, note, transaction_date) 
                            VALUES (" . $account_id . ", " . $category_id . ", -" . $amount_clean . ", '" . $mysql->real_escape_string($note_text) . "', '" . $transaction_date . "')";

                    if($mysql->query($sql)) {
                        $message = 'Expense added successfully!';
                        $message_type = 'success';
                        // Clear form by redirecting
                        header('Location: add_expense.php?success=1');
                        exit();
                    } else {
                        $message = 'Error adding expense: ' . $mysql->error;
                        $message_type = 'error';
                    }
                } else {
                    $message = 'Invalid account selected.';
                    $message_type = 'error';
                }
            }
        }
    }
}

// Check for success message
if(isset($_GET['success']) && $_GET['success'] == 1) {
    $message = 'Expense added successfully!';
    $message_type = 'success';
}

// Get expense categories for dropdown
$expense_categories_sql = "SELECT category_id, label FROM categories WHERE type = 'expense' AND (user_id = " . $user_id . " OR user_id IS NULL) ORDER BY label";
$expense_categories_result = $mysql->query($expense_categories_sql);
$expense_categories = [];
if($expense_categories_result) {
    while($row = $expense_categories_result->fetch_assoc()) {
        $expense_categories[] = $row;
    }
}

// Get user's accounts
$accounts_sql = "SELECT account_id, name FROM accounts WHERE user_id = " . $user_id . " ORDER BY name";
$accounts_result = $mysql->query($accounts_sql);
$accounts = [];
if($accounts_result) {
    while($row = $accounts_result->fetch_assoc()) {
        $accounts[] = $row;
    }
}

// Get recent expense transactions
$recent_expense_sql = "SELECT t.note, t.amount, t.transaction_date, c.label as category_name 
                      FROM transactions t
                      LEFT JOIN categories c ON t.category_id = c.category_id
                      WHERE c.type = 'expense' 
                        AND t.account_id IN (
                            SELECT account_id FROM accounts WHERE user_id = " . $user_id . "
                            UNION
                            SELECT account_id FROM shared_users WHERE user_id = " . $user_id . "
                        )
                      ORDER BY t.transaction_date DESC, t.transaction_id DESC
                      LIMIT 4";
$recent_expense_result = $mysql->query($recent_expense_sql);
$recent_expense = [];
if($recent_expense_result) {
    while($row = $recent_expense_result->fetch_assoc()) {
        $recent_expense[] = $row;
    }
}

// Get expense summary for current month
$current_month_start = date('Y-m-01');
$expense_summary_sql = "SELECT 
                        COUNT(*) as expense_count,
                        SUM(ABS(t.amount)) as total_expense,
                        AVG(ABS(t.amount)) as avg_expense
                       FROM transactions t
                       LEFT JOIN categories c ON t.category_id = c.category_id
                       WHERE c.type = 'expense' 
                         AND t.transaction_date >= '" . $current_month_start . "'
                         AND t.account_id IN (
                             SELECT account_id FROM accounts WHERE user_id = " . $user_id . "
                             UNION
                             SELECT account_id FROM shared_users WHERE user_id = " . $user_id . "
                         )";
$expense_summary_result = $mysql->query($expense_summary_sql);
$expense_summary = ['expense_count' => 0, 'total_expense' => 0, 'avg_expense' => 0];
if($expense_summary_result && $expense_summary_result->num_rows > 0) {
    $expense_summary = $expense_summary_result->fetch_assoc();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <!-- Google tag (gtag.js) -->
    <script async src="https://www.googletagmanager.com/gtag/js?id=G-P8J7Y3RS27"></script>
    <script>
      window.dataLayer = window.dataLayer || [];
      function gtag(){dataLayer.push(arguments);}
      gtag('js', new Date());
      gtag('config', 'G-P8J7Y3RS27');
    </script>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Expense - Budget Buddy</title>
    <link rel="stylesheet" href="css/style.css?v=<?php echo time(); ?>">
</head>
<body class="expense-page">
<?php
$active_page = 'add_expense';
include 'sidebar.php';
?>

<main class="main-content">
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Add Expense</h1>
        <p class="page-subtitle">Upload receipts and track your spending</p>
    </div>

    <?php if($message): ?>
        <div class="message <?php echo $message_type; ?>" style="padding: 12px; margin-bottom: 20px; border-radius: 6px; <?php echo $message_type == 'success' ? 'background-color: #ffebee; color: #c62828; border: 1px solid #d32f2f;' : 'background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb;'; ?>">
            <?php echo htmlspecialchars($message); ?>
        </div>
    <?php endif; ?>

    <div class="content-layout" style="align-items: start;">
        <!-- Left Column: Expense Details -->
        <div class="form-section">
            <!-- Toggle Button -->
            <div style="display: flex; justify-content: flex-end; margin-bottom: 20px; gap: 10px; align-items: center;">
                <span style="font-size: 14px; color: #666;">Manual Input</span>
                <label style="position: relative; display: inline-block; width: 50px; height: 26px;">
                    <input type="checkbox" id="input-mode-toggle" style="opacity: 0; width: 0; height: 0;">
                    <span style="position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0; background-color: #ccc; transition: .4s; border-radius: 26px;"></span>
                    <span style="position: absolute; content: ''; height: 20px; width: 20px; left: 3px; bottom: 3px; background-color: white; transition: .4s; border-radius: 50%;"></span>
                </label>
                <span style="font-size: 14px; color: #666;">📷 Image Upload</span>
            </div>

            <!-- Manual Input Form -->
            <div id="manual-input-mode">
                <form id="manual-expense-form" method="POST" action="add_expense.php" class="expense-form">
                    <input type="hidden" name="mode" value="manual">
                    <div class="form-field">
                        <label for="merchant-name">Merchant Name</label>
                        <input type="text" id="merchant-name" name="merchant-name" placeholder="e.g., Walmart" value="<?php echo htmlspecialchars($prefill_merchant); ?>" required>
                    </div>

                    <div class="form-field">
                        <label for="expense-amount">Amount</label>
                        <input type="text" id="expense-amount" name="expense-amount" class="expense-amount-input" placeholder="$ 0.00" value="<?php echo htmlspecialchars($prefill_amount); ?>" required>
                    </div>

                    <div class="form-field">
                        <label for="expense-category-input">Category</label>
                        <input type="text" id="expense-category-input" name="expense-category-input" list="expense-category-list" placeholder="Select or type a category name" autocomplete="off" value="<?php echo htmlspecialchars($prefill_category); ?>" required>
                        <datalist id="expense-category-list">
                            <?php foreach($expense_categories as $cat): ?>
                            <option value="<?php echo htmlspecialchars($cat['label']); ?>">
                                <?php endforeach; ?>
                        </datalist>
                    </div>

                    <div class="form-field">
                        <label for="expense-account">Account</label>
                        <select id="expense-account" name="expense-account" required>
                            <option value="">Choose your bank account</option>
                            <?php foreach($accounts as $acc): ?>
                                <option value="<?php echo $acc['account_id']; ?>" <?php echo ($prefill_account == $acc['account_id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars($acc['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-field">
                        <label for="expense-date">Date Spent</label>
                        <div class="date-input-wrapper">
                            <input type="date" id="expense-date" name="expense-date" value="<?php echo htmlspecialchars($prefill_date ?: date('Y-m-d')); ?>" required>
                            <span class="date-icon">📅</span>
                        </div>
                    </div>

                    <div class="form-field">
                        <label for="expense-notes">Notes (Optional)</label>
                        <textarea id="expense-notes" name="expense-notes" placeholder="Add any additional notes..."><?php echo htmlspecialchars($prefill_notes); ?></textarea>
                    </div>

                    <button type="submit" class="expense-submit-button">
                        <span>✓</span>
                        <span>Add Expense</span>
                    </button>
                </form>
            </div>

            <!-- Image Upload Mode -->
            <div id="image-upload-mode" style="display: none;">
                <div class="upload-section" style="padding: 40px; max-width: 600px; margin: 0 auto;">
                    <div class="upload-header" style="margin-bottom: 24px; text-align: center;">
                        <h2 class="card-title-section" style="font-size: 20px; margin-bottom: 8px;">Upload Receipt Image</h2>
                        <p class="card-subtitle-section" style="font-size: 14px; color: #666;">Upload a photo of your receipt to automatically extract expense details</p>
                    </div>

                    <form method="POST" action="add_expense.php" enctype="multipart/form-data" style="display:flex; flex-direction:column; gap:16px;">
                        <input type="hidden" name="mode" value="upload">
                        <div class="upload-area" style="padding: 40px 20px; margin-bottom: 8px; border: 2px dashed #ccc; text-align: center; border-radius: 10px; cursor: pointer;">
                            <span class="upload-icon" style="font-size: 48px; margin-bottom: 12px; display:block;">📄</span>
                            <div class="upload-text" style="font-size: 16px; margin-bottom: 4px;">Drag & drop or browse</div>
                            <div class="upload-hint" style="font-size: 12px; color: #999;">PNG, JPG, PDF (max 10MB)</div>
                            <input type="file" name="expense-upload" accept=".png,.jpg,.jpeg,.pdf" style="margin-top:12px;" required>
                        </div>

                        <button type="submit" class="expense-submit-button" style="justify-content:center;">
                            <span>✨</span>
                            <span>Extract & Prefill</span>
                        </button>
                    </form>
                </div>
            </div>
        </div>

        <!-- Right Column: Recent Expenses -->
        <div class="sidebar-section">
            <!-- Recent Expenses Card -->
            <div class="recent-income-card">
                <div class="card-header-section">
                    <div class="card-title-section">Recent Expenses</div>
                    <div class="card-subtitle-section">Your latest expense entries</div>
                </div>

                <div class="income-list">
                    <?php if(count($recent_expense) > 0): ?>
                        <?php foreach($recent_expense as $expense): ?>
                            <div class="income-item">
                                <div class="income-item-info">
                                    <div class="income-item-title"><?php echo htmlspecialchars($expense['note'] ?? 'No note'); ?></div>
                                    <div class="income-item-meta"><?php echo date('Y-m-d', strtotime($expense['transaction_date'])); ?> • <?php echo htmlspecialchars($expense['category_name'] ?? 'Uncategorized'); ?></div>
                                </div>
                                <div class="income-item-amount" style="color: #d32f2f;">-$<?php echo number_format(abs($expense['amount']), 2); ?></div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div style="padding: 20px; text-align: center; color: #666;">
                            <p>No recent expense entries</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Expense Summary Card -->
            <div class="summary-card" style="margin-top: 20px;">
                <div class="card-header-section">
                    <div class="card-title-section">Expense Summary</div>
                    <div class="card-subtitle-section">This month</div>
                </div>

                <div class="summary-stats">
                    <div class="summary-stat">
                        <span class="summary-stat-label">Total Expenses</span>
                        <span class="summary-stat-value" style="color: #d32f2f;">-$<?php echo number_format($expense_summary['total_expense'] ?? 0, 2); ?></span>
                    </div>

                    <div class="summary-stat">
                        <span class="summary-stat-label">Expense Count</span>
                        <span class="summary-stat-value"><?php echo $expense_summary['expense_count'] ?? 0; ?></span>
                    </div>

                    <div class="summary-stat">
                        <span class="summary-stat-label">Average per Expense</span>
                        <span class="summary-stat-value">$<?php echo number_format($expense_summary['avg_expense'] ?? 0, 2); ?></span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</main>
<script>
    // Toggle between manual input and image upload modes
    const toggle = document.getElementById('input-mode-toggle');
    const manualMode = document.getElementById('manual-input-mode');
    const imageMode = document.getElementById('image-upload-mode');
    const toggleSlider = toggle.nextElementSibling;
    const toggleCircle = toggleSlider.nextElementSibling;

    toggle.addEventListener('change', function() {
        if(this.checked) {
            // Switch to image upload mode
            manualMode.style.display = 'none';
            imageMode.style.display = 'block';
            toggleSlider.style.backgroundColor = '#d32f2f';
            toggleCircle.style.transform = 'translateX(24px)';
        } else {
            // Switch to manual input mode
            manualMode.style.display = 'block';
            imageMode.style.display = 'none';
            toggleSlider.style.backgroundColor = '#ccc';
            toggleCircle.style.transform = 'translateX(0)';
        }
    });

    // If we just extracted data, force manual mode to review
    const forceManualExpense = <?php echo $force_manual_mode ? 'true' : 'false'; ?>;
    if(forceManualExpense) {
        toggle.checked = false;
        manualMode.style.display = 'block';
        imageMode.style.display = 'none';
        toggleSlider.style.backgroundColor = '#ccc';
        toggleCircle.style.transform = 'translateX(0)';
    }

    // Ensure form validates category input before submission
    const manualExpenseForm = document.getElementById('manual-expense-form');
    if(manualExpenseForm){
        manualExpenseForm.addEventListener('submit', function(e) {
            const categoryInput = document.getElementById('expense-category-input');
            if(!categoryInput.value || categoryInput.value.trim() === '') {
                e.preventDefault();
                alert('Please select or enter a category');
                categoryInput.focus();
                return false;
            }
        });
    }
</script>
<style>
    #input-mode-toggle:checked + span {
        background-color: #d32f2f !important;
    }
    #input-mode-toggle:checked + span + span {
        transform: translateX(24px);
    }
</style>
</body>
</html>
<?php
$mysql->close();
?>
