<?php
require_once 'config.php';
include 'security.php';

// Get the logged-in user's ID
$user_id = isset($_SESSION['user_id']) ? intval($_SESSION['user_id']) : 0;

if($user_id <= 0) {
    header('Location: login.php');
    exit();
}

$message = '';
$message_type = '';

// Handle form submission
if($_SERVER['REQUEST_METHOD'] == 'POST') {
    $account_name = isset($_POST['account-name']) ? trim($_POST['account-name']) : '';
    $account_type = isset($_POST['account-type']) ? trim($_POST['account-type']) : '';

    // Validate required fields
    if(empty($account_name)) {
        $message = 'Please enter an account name.';
        $message_type = 'error';
    } else {
        // Insert account (only using columns that exist: user_id, name)
        // If type column exists, include it; otherwise just use name
        $sql = "INSERT INTO accounts (user_id, name) 
                VALUES (" . $user_id . ", '" . $mysql->real_escape_string($account_name) . "')";

        if($mysql->query($sql)) {
            $message = 'Account added successfully!';
            $message_type = 'success';
            // Clear form by redirecting
            header('Location: add_account.php?success=1');
            exit();
        } else {
            $message = 'Error adding account: ' . $mysql->error;
            $message_type = 'error';
        }
    }
}

// Handle account deletion
if(isset($_GET['delete']) && !empty($_GET['delete'])) {
    $account_id_to_delete = intval($_GET['delete']);
    
    // Verify the account belongs to the user
    $verify_sql = "SELECT account_id FROM accounts WHERE account_id = " . $account_id_to_delete . " AND user_id = " . $user_id;
    $verify_result = $mysql->query($verify_sql);
    
    if($verify_result && $verify_result->num_rows > 0) {
        // Start transaction to ensure both deletions succeed or fail together
        $mysql->begin_transaction();
        
        try {
            // First, delete all associated transactions
            $delete_transactions_sql = "DELETE FROM transactions WHERE account_id = " . $account_id_to_delete;
            if(!$mysql->query($delete_transactions_sql)) {
                throw new Exception('Error deleting transactions: ' . $mysql->error);
            }
            
            // Also delete from shared_users if the account is shared
            $delete_shared_sql = "DELETE FROM shared_users WHERE account_id = " . $account_id_to_delete;
            $mysql->query($delete_shared_sql); // This might fail if table doesn't exist, so we don't check
            
            // Then delete the account
            $delete_account_sql = "DELETE FROM accounts WHERE account_id = " . $account_id_to_delete . " AND user_id = " . $user_id;
            if(!$mysql->query($delete_account_sql)) {
                throw new Exception('Error deleting account: ' . $mysql->error);
            }
            
            // Commit the transaction
            $mysql->commit();
            $message = 'Account and all associated transactions deleted successfully!';
            $message_type = 'success';
            header('Location: add_account.php?success=deleted');
            exit();
        } catch (Exception $e) {
            // Rollback on error
            $mysql->rollback();
            $message = 'Error deleting account: ' . $e->getMessage();
            $message_type = 'error';
        }
    } else {
        $message = 'Account not found or you do not have permission to delete it.';
        $message_type = 'error';
    }
}

// Check for success message
if(isset($_GET['success']) && $_GET['success'] == 1) {
    $message = 'Account added successfully!';
    $message_type = 'success';
} elseif(isset($_GET['success']) && $_GET['success'] == 'deleted') {
    $message = 'Account and all associated transactions deleted successfully!';
    $message_type = 'success';
}

// Get user's existing accounts for display
$accounts_sql = "SELECT account_id, name FROM accounts WHERE user_id = " . $user_id . " ORDER BY name";
$accounts_result = $mysql->query($accounts_sql);
$accounts = [];
if($accounts_result) {
    while($row = $accounts_result->fetch_assoc()) {
        // Calculate balance for this account
        $account_balance_sql = "SELECT COALESCE(SUM(amount), 0) AS account_balance 
                                FROM transactions 
                                WHERE account_id = " . intval($row['account_id']);
        $account_balance_result = $mysql->query($account_balance_sql);
        $account_balance = 0;
        if($account_balance_result && $account_balance_result->num_rows > 0) {
            $balance_row = $account_balance_result->fetch_assoc();
            $account_balance = floatval($balance_row['account_balance'] ?? 0);
        }
        $row['balance'] = $account_balance;
        $accounts[] = $row;
    }
}

// Calculate totals
$total_accounts = count($accounts);

// Calculate total balance from all transactions (assisted by AI)
$balance_sql = "
    SELECT COALESCE(SUM(t.amount), 0) AS total_balance
    FROM transactions t
    WHERE t.account_id IN (
        SELECT account_id FROM accounts WHERE user_id = " . $user_id . "
        UNION
        SELECT account_id FROM shared_users WHERE user_id = " . $user_id . "
    )
";
$balance_result = $mysql->query($balance_sql);
$total_balance = 0;
if($balance_result && $balance_result->num_rows > 0) {
    $balance_row = $balance_result->fetch_assoc();
    $total_balance = floatval($balance_row['total_balance'] ?? 0);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <!-- Google tag (gtag.js) -->
    <script async src="https://www.googletagmanager.com/gtag/js?id=G-P8J7Y3RS27"></script>
    <script>
      window.dataLayer = window.dataLayer || [];
      function gtag(){dataLayer.push(arguments);}
      gtag('js', new Date());
      gtag('config', 'G-P8J7Y3RS27');
    </script>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Account - Budget Buddy</title>
    <link rel="stylesheet" href="css/style.css?v=<?php echo time(); ?>">
</head>
<body class="account-page">
<?php
$active_page = 'add_account';
include 'sidebar.php';
?>

<main class="main-content">
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Add Account</h1>
        <p class="page-subtitle">Create a new financial account</p>
    </div>

    <div class="content-layout">
        <!-- Left Column: Add Account Form -->
        <div class="form-section">
            <?php if($message): ?>
                <div class="message <?php echo $message_type; ?>" style="padding: 12px; margin-bottom: 20px; border-radius: 6px; <?php echo $message_type == 'success' ? 'background-color: #f3e5f5; color: #6a1b9a; border: 1px solid #9c27b0;' : 'background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb;'; ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <!-- Account Form -->
            <form method="POST" action="add_account.php" class="income-form">
                <div class="form-field">
                    <label for="account-name">Account Name</label>
                    <input type="text" id="account-name" name="account-name" placeholder="e.g., Main Checking, Savings Account" required>
                </div>

                <div class="form-field">
                    <label for="account-type">Account Type (Optional)</label>
                    <select id="account-type" name="account-type">
                        <option value="">Select account type</option>
                        <option value="checking">Checking</option>
                        <option value="savings">Savings</option>
                        <option value="credit_card">Credit Card</option>
                        <option value="investment">Investment</option>
                        <option value="cash">Cash</option>
                        <option value="other">Other</option>
                    </select>
                </div>

                <button type="submit" class="account-submit-button">
                    <span>✓</span>
                    <span>Add Account</span>
                </button>
            </form>
        </div>

        <!-- Right Column: Account Summary -->
        <div class="sidebar-section">
            <!-- Summary Cards -->
            <div class="summary-card">
                <div class="card-header-section">
                    <div class="card-title-section">Total Balance</div>
                    <div class="card-subtitle-section">Across all accounts</div>
                </div>
                <div class="summary-stats">
                    <div class="summary-stat" style="border-bottom: none;">
                        <span class="summary-stat-value" style="font-size: 32px; color: #9c27b0; font-weight: 600;">$<?php echo number_format($total_balance, 2); ?></span>
                        <div style="font-size: 12px; color: #999; margin-top: 4px;">(Calculated from transactions)</div>
                    </div>
                </div>
            </div>

            <!-- Recent Accounts List -->
            <div class="recent-income-card">
                <div class="card-header-section">
                    <div class="card-title-section">Your Accounts <span style="color: #9c27b0; font-weight: 600;">(<?php echo $total_accounts; ?>)</span></div>
                    <div class="card-subtitle-section">All financial accounts</div>
                </div>

                <div class="income-list">
                    <?php if(count($accounts) > 0): ?>
                        <?php foreach($accounts as $acc): ?>
                            <div class="income-item" style="position: relative;">
                                <div class="income-item-info">
                                    <div class="income-item-title"><?php echo htmlspecialchars($acc['name']); ?></div>
                                    <div class="income-item-meta">Account ID: <?php echo $acc['account_id']; ?></div>
                                </div>
                                <div style="display: flex; align-items: center; gap: 10px;">
                                    <div class="income-item-amount" style="color: #9c27b0; font-weight: 600;">
                                        $<?php echo number_format($acc['balance'], 2); ?>
                                    </div>
                                    <button type="button" 
                                            class="delete-account-btn" 
                                            data-account-id="<?php echo $acc['account_id']; ?>"
                                            data-account-name="<?php echo htmlspecialchars($acc['name']); ?>"
                                            style="background: #d32f2f; color: white; border: none; border-radius: 4px; padding: 6px 12px; cursor: pointer; font-size: 12px; transition: background 0.2s;"
                                            onmouseover="this.style.background='#c62828'"
                                            onmouseout="this.style.background='#d32f2f'">
                                        🗑️ Delete
                                    </button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div style="padding: 20px; text-align: center; color: #666;">
                            <p>No accounts yet</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</main>

<!-- Confirmation Modal -->
<div id="deleteModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000; align-items: center; justify-content: center;">
    <div style="background: white; padding: 30px; border-radius: 8px; max-width: 400px; box-shadow: 0 4px 6px rgba(0,0,0,0.1);">
        <h3 style="margin: 0 0 15px 0; color: #333; font-size: 20px;">Confirm Deletion</h3>
        <p id="deleteModalMessage" style="margin: 0 0 20px 0; color: #666; line-height: 1.5;"></p>
        <div style="display: flex; gap: 10px; justify-content: flex-end;">
            <button id="cancelDeleteBtn" style="padding: 10px 20px; background: #e0e0e0; border: none; border-radius: 4px; cursor: pointer; font-size: 14px;">Cancel</button>
            <button id="confirmDeleteBtn" style="padding: 10px 20px; background: #d32f2f; color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 14px;">Delete</button>
        </div>
    </div>
</div>

<script>
    // Handle delete account button clicks
    document.addEventListener('DOMContentLoaded', function() {
        const deleteButtons = document.querySelectorAll('.delete-account-btn');
        const modal = document.getElementById('deleteModal');
        const modalMessage = document.getElementById('deleteModalMessage');
        const confirmBtn = document.getElementById('confirmDeleteBtn');
        const cancelBtn = document.getElementById('cancelDeleteBtn');
        let accountIdToDelete = null;
        let accountNameToDelete = '';

        deleteButtons.forEach(button => {
            button.addEventListener('click', function() {
                accountIdToDelete = this.getAttribute('data-account-id');
                accountNameToDelete = this.getAttribute('data-account-name');
                modalMessage.textContent = 'Are you sure you want to delete the account "' + accountNameToDelete + '"? All associated transactions will also be deleted.';
                modal.style.display = 'flex';
            });
        });

        cancelBtn.addEventListener('click', function() {
            modal.style.display = 'none';
            accountIdToDelete = null;
        });

        confirmBtn.addEventListener('click', function() {
            if(accountIdToDelete) {
                window.location.href = 'add_account.php?delete=' + accountIdToDelete;
            }
        });

        // Close modal when clicking outside
        modal.addEventListener('click', function(e) {
            if(e.target === modal) {
                modal.style.display = 'none';
                accountIdToDelete = null;
            }
        });
    });
</script>

</body>
</html>
<?php
$mysql->close();
?>
